declare namespace global {
    let behaviorSystem: BehaviorSystem;
}

@typedef
export class CustomFunctions {
    @input
    script: ScriptComponent;

    @input
    function: string;
}

export enum CallbackType {
    None,
    Behavior,
    BehaviorCustom,
    CustomFunction
}

interface ScriptWithCallbackInputs extends BaseScriptComponent{
    callbackType: CallbackType
}

interface BehaviorScript {
    trigger: () => void
}

export class BehaviorEventCallbacks {

    static invokeCallbackFromInputs(script: ScriptWithCallbackInputs, eventName: string): (data: any) => void {
        const callbackType: CallbackType = script.callbackType;
        if (isNull(callbackType)) {
            print("Warning: Wrong Callback Type input.");
        }
        switch (callbackType) {
            case CallbackType.Behavior:
                return BehaviorEventCallbacks.invokeCallbackFromBehavior(script, eventName);
            case CallbackType.BehaviorCustom:
                return BehaviorEventCallbacks.invokeCallbackFromBehaviorSystem(script, eventName);
            case CallbackType.CustomFunction:
                return BehaviorEventCallbacks.invokeCallbackFromScriptAPI(script, eventName);
        }
    }

    private static invokeCallbackFromBehavior(script: BaseScriptComponent, eventName: string): (data: any) => void {
        return () => {
            const behaviors: BehaviorScript[] = script[eventName + "Behaviors"];
            if (!behaviors) {
                print("Warning: No event with name: " + eventName);
                return;
            }
            for (const behavior of behaviors) {
                behavior?.trigger?.();
            }
        };
    }

    private static invokeCallbackFromBehaviorSystem(script: BaseScriptComponent, eventName: string): (data: any) => void {
        return () => {
            if (!global.behaviorSystem) {
                print("The global behavior system has not been instantiated yet! Make sure a Behavior script is present somewhere!");
                return;
            }
            const triggerNames: string[] = script[eventName + "CustomTriggers"];
            for (const triggerName of triggerNames) {
                if (triggerName.length == 0) {
                    print("You are trying to send an empty string custom trigger!");
                    continue;
                }
                global.behaviorSystem.sendCustomTrigger(triggerName);
            }
        };
    }

    private static invokeCallbackFromScriptAPI(script: BaseScriptComponent, eventName: string): (data: any) => void {
        return (eventData: any) => {
            const listeners: CustomFunctions[] = script[eventName + "Functions"];
            if (!listeners) {
                return;
            }
            for (const listener of listeners) {
                if (listener.function.length == 0) {
                    print("You are trying to invoke an empty string function!");
                    continue;
                }
                if (!listener.script[listener.function]) {
                    print("Cannot find " + listener.function + " function in the assigned " + listener.script.getSceneObject().name + " script.");
                    continue;
                } else if (typeof listener.script[listener.function] !== "function") {
                    print("There is a " + listener.function + " property in the assigned " + listener.script.getSceneObject().name + " script, but it's not a function.");
                    continue;
                }

                listener.script[listener.function](eventData);
            }
        };
    }
}
